<?php
class ModelExtensionModulePromotions extends Model
{
    private $module = array();

    public function __construct($registry)
    {
        parent::__construct($registry);

        $this->config->load('isenselabs/promotions');
        $this->module = $this->config->get('promotions');

        $this->load->model('setting/setting');

        // Module setting
        $setting = $this->model_setting_setting->getSetting($this->module['code'], $this->config->get('config_store_id'));
        $this->module['setting'] = array_replace_recursive(
            $this->module['setting'],
            !empty($setting[$this->module['code'] . '_setting']) ? $setting[$this->module['code'] . '_setting'] : array()
        );

        // Misc
        $this->module['page_seo_url'] = $this->module['setting']['page']['seo_url'][0];
        if ($this->config->get('config_seo_url') && !empty($this->module['setting']['page']['seo_url'][$this->config->get('config_language_id')])) {
            $this->module['page_seo_url'] = $this->module['setting']['page']['seo_url'][$this->config->get('config_language_id')];
        }
    }

    /**
     * Responsible to fetch promotions rules
     *
     * @return array
     */
    public function getPromotions($queries = array())
    {
        $items = array();
        $queries = array_merge(
            array(
                'where' => " ",
                'order' => " ORDER BY priority_order ASC",
                'limit' => " LIMIT 0, 1000"
            ),
            $queries
        );

        $results = $this->db->query(
            "SELECT pi.*, IF (pi.priority > 0, pi.priority, 99999) as priority_order
            FROM `" . DB_PREFIX . "promotions_item` pi
            WHERE " . $this->promoValidationQuery() . $queries['where']
            . $queries['order']
            . $queries['limit']
        );

        foreach ($results->rows as $promotion) {
            $items[] = $this->extractPromotions($promotion);
        }

        return $items;
    }

    public function getPromotion($promo_id)
    {
        $result = $this->db->query(
            "SELECT pi.*
            FROM `" . DB_PREFIX . "promotions_item` pi
            WHERE pi.promotion_id = " . (int)$promo_id . "
                AND " . $this->promoValidationQuery()
        );

        if ($result->row) {
            return $this->extractPromotions($result->row);
        }

        return false;
    }

    /**
     * Promotion rule validation in query:
     * - Check promotion status
     * - Check stores
     * - Check customer groups
     * - Check customer profile
     * - Check start and end date
     * - Check limit usages
     * - Put priority order 0 at the end of results
     */
    private function promoValidationQuery()
    {
        $customer_id = $this->customer->getId() ? $this->customer->getId() : 0;
        $customer_group_id = $this->customer->getGroupId() ? $this->customer->getGroupId() : 0;

        return " pi.status = 1
                AND pi.stores LIKE '%\"" . (int)$this->config->get('config_store_id') . "\"%'
                AND IF (pi.limit_customer_groups = 1, pi.customer_group_ids LIKE '%\"" . (int)$customer_group_id . "\"%', TRUE)
                AND IF (pi.limit_customer_profile = 1, pi.customer_ids LIKE '%\"" . (int)$customer_id . "\"%', TRUE)
                AND CURDATE() >= pi.date_start
                AND (pi.date_end >= CURDATE() or pi.date_end = 0)
                AND IF (pi.limit_usage = 1, pi.limit_max_usage > (SELECT COUNT(*) FROM `" . DB_PREFIX . "promotions_log` pl WHERE pl.promotion_id = pi.promotion_id), TRUE) ";
    }

    public function getProducts($queries = array())
    {
        $products = array();
        $queries = array_merge(
            array(
                'join'  => " ",
                'where' => " ",
                'order' => " ORDER BY p.date_added DESC",
                'limit' => " LIMIT 0, 1000"
            ),
            $queries
        );

        // Fetch products
        $query = $this->db->query("SELECT p.product_id, (SELECT AVG(rating) AS total FROM " . DB_PREFIX . "review r1 WHERE r1.product_id = p.product_id AND r1.status = '1' GROUP BY r1.product_id) AS rating, (SELECT price FROM " . DB_PREFIX . "product_discount pd2 WHERE pd2.product_id = p.product_id AND pd2.customer_group_id = '" . (int)$this->config->get('config_customer_group_id') . "' AND pd2.quantity = '1' AND ((pd2.date_start = '0000-00-00' OR pd2.date_start < NOW()) AND (pd2.date_end = '0000-00-00' OR pd2.date_end > NOW())) ORDER BY pd2.priority ASC, pd2.price ASC LIMIT 1) AS discount, (SELECT price FROM " . DB_PREFIX . "product_special ps WHERE ps.product_id = p.product_id AND ps.customer_group_id = '" . (int)$this->config->get('config_customer_group_id') . "' AND ((ps.date_start = '0000-00-00' OR ps.date_start < NOW()) AND (ps.date_end = '0000-00-00' OR ps.date_end > NOW())) ORDER BY ps.priority ASC, ps.price ASC LIMIT 1) AS special
            FROM " . DB_PREFIX . "product p
                LEFT JOIN " . DB_PREFIX . "product_description pd ON (p.product_id = pd.product_id)
                LEFT JOIN " . DB_PREFIX . "product_to_store p2s ON (p.product_id = p2s.product_id)
                " . $queries['join'] . "
            WHERE pd.language_id = '" . (int)$this->config->get('config_language_id') . "'
                AND p.status = '1'
                AND p.date_available <= NOW()
                AND p2s.store_id = '" . (int)$this->config->get('config_store_id') . "'
                " . $queries['where'] . "
            GROUP BY p.product_id"
            . $queries['order']
            . $queries['limit']);

        // Products detail
        $results = array();
        if ($query->rows) {
            $this->load->model('catalog/product');
            foreach ($query->rows as $result) {
                $results[$result['product_id']] = $this->model_catalog_product->getProduct($result['product_id']);
            }
        }

        // Products variables
        if ($results) {
            $this->load->model('tool/image');

            $image_width = $image_height = 240;
            $image = $this->model_tool_image->resize('placeholder.png', $image_width, $image_height);

            foreach ($results as $result) {
                if ($result['image']) {
                    $image = $this->model_tool_image->resize($result['image'], $image_width, $image_height);
                }

                $price = false;
                if ($this->customer->isLogged() || !$this->config->get('config_customer_price')) {
                    $price = $this->currency->format($this->tax->calculate($result['price'], $result['tax_class_id'], $this->config->get('config_tax')), $this->session->data['currency']);
                }

                $special = false;
                if ((float)$result['special']) {
                    $special = $this->currency->format($this->tax->calculate($result['special'], $result['tax_class_id'], $this->config->get('config_tax')), $this->session->data['currency']);
                }

                $tax = false;
                if ($this->config->get('config_tax')) {
                    $tax = $this->currency->format((float)$result['special'] ? $result['special'] : $result['price'], $this->session->data['currency']);
                }

                $rating = false;
                if ($this->config->get('config_review_status')) {
                    $rating = $result['rating'];
                }

                $products[] = array(
                    'product_id'  => $result['product_id'],
                    'thumb'       => $image,
                    'name'        => $result['name'],
                    'price'       => $price,
                    'special'     => $special,
                    'tax'         => $tax,
                    'rating'      => $rating,
                    'href'        => $this->url->link('product/product', 'product_id=' . $result['product_id'])
                );
            }
        }

        return $products;
    }

    public function couponCheck($code)
    {
        if ($this->module['setting']['status']) {
            $result = $this->db->query("SELECT pi.coupon_code
                FROM `" . DB_PREFIX . "promotions_item` pi
                WHERE pi.coupon_code = '" . $this->db->escape($code) . "'
                AND " . $this->promoValidationQuery()
            );

            if ($result->num_rows) {
                return true;
            }
        }

        return false;
    }

    /**
     * Responsible to standarize result format
     *
     * @param  array   $itemValues      Single promotion rule
     * @param  boolean $specificLang    Automatically pick based on store lang-id
     *
     * @return array
     */
    protected function extractPromotions($itemValues, $specificLang = true)
    {
        $item    = $itemValues;
        $lang_id = $this->config->get('config_language_id');
        $titles  = array('title', 'design_page_message', 'message_congrats', 'message_eligible', 'message_upsell');
        $decodes = array('condition_min_quantities', 'condition_min_amounts', 'condition_product_ids', 'condition_category_ids', 'condition_manufacturer_ids', 'discount_product_ids',
                        'discount_category_ids', 'discount_manufacturer_ids', 'discount_values', 'zone_ids', 'excluded_category_ids', 'customer_group_ids', 'customer_ids', 'stores', 'meta');

        foreach ($itemValues as $key => $value) {
            $temp_val = json_decode($value, true);

            if (!is_array($temp_val)) {
                continue;
            }

            if (in_array($key, $titles) || in_array($key, $decodes)) {
                $item[$key] = $temp_val;

                if (!empty($temp_val) && $specificLang && in_array($key, $titles)) {
                    $item[$key] = isset($temp_val[$lang_id]) ? $temp_val[$lang_id] : $temp_val[array_rand($temp_val)];
                }
            }
        }

        if ($item) {
            $item['page_url'] = $this->urlPromo($item['promotion_id']);
        }

        return array_replace_recursive($this->module['setting']['item'], $item);
    }

    /**
     * Shortcut to create promotion url
     *
     * @param  int $promo_id
     *
     * @return string
     */
    protected function urlPromo($promo_id)
    {
        return $this->config->get('config_seo_url') && $this->module['page_seo_url'] ? $this->module['page_seo_url'] . '?promo_id=' . (int)$promo_id : $this->url->link($this->module['path'] . '/page', 'promo_id=' . (int)$promo_id, true);
    }
}
